unit uPHPInspector;

{
  PHP Inspector
  -----------------------------------------------------------------------------
  PHP Inspection Unit for Borland Delphi
    (c) 2004 - 2005 by David Fekete
    http://www.mirage228.net,
    mirage228@gmx.net
    All Rights reserved.
  -----------------------------------------------------------------------------
  Licence:

  The contents of this file are subject to the Mozilla Public License Version
  1.1 (the "License"); you may not use this file except in compliance with the
  License. You may obtain a copy of the License at http://www.mozilla.org/MPL/

  Software distributed under the License is distributed on an "AS IS" basis,
  WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
  the specific language governing rights and limitations under the License.

  The Original Code is: uPHPInspector.pas, released on November 1 2004

  The Initial Developer of the Original Code is David Fekete.
  Portions created by David Fekete are Copyright (c) 2004 - 2005 David Fekete.
  All Rights Reserved.

  -----------------------------------------------------------------------------
  PCRE 5.0 Wrapper for Delphi 7 (c) 2003 - 2004 by Renato Mancuso,
    http://www.renatomancuso.com 
    All rights reserved.
  -----------------------------------------------------------------------------
  PCRE 5.0 by
    Philip Hazel <ph10@cam.ac.uk>
    University of Cambridge Computing Service,
    Cambridge, England. Phone: +44 1223 334714.

    Copyright (c) 1997-2004 University of Cambridge
    All rights reserved.
  -----------------------------------------------------------------------------
  Requirements:
    Delphi 6 Personal or higher (because of TObjectList, Unit Contnrs)
      Successfully tested with:
        - Borland Delphi 6 Personal
        - Borland Delphi 7 Professional
        - Borland Delphi 2005 Professional (VCL for Win32)
        
    PCRE 5.0 Wrapper Units for Delphi (included) and the PCRE 5.0 DLL
      (pcre.dll, included)
  -----------------------------------------------------------------------------
  Version History:

  Version 2.5 (Oct 03 2005)
    - Fixed error in comment detection which was caused by an \" or \' escape
      element embedded in a single or double quote. That caused the parser
      to stop parsing at this point
    - Fixed error when detecting include files which had no semicolon or
      line-end after the filename (e.g. <? include "abc.htm" ?>)
    - Added support for Unix-Shell style comments (# <comment>)
    - Added new compiler switch "NODUPLICATES" (off by default).
      If defined the parser won't add the same class, interfac
      or method to the result lists more than one time.

  Version 2.4 (May 23 2005)
    - Fixed wrong entity positions when there was text before the first php-tag
    - Fixed wrong entity positions when there were more than one php-tag pairs
    - Some speed optimizations to the FindEntityAt() routine

  Version 2.3 (March 27 2005)
    - Include files will not be recognized anymore when there is no include
      file given (e.g. when you just write "include" w/o filename)
    - Fixed error that variables were errorneusly recognized as include files
      (e.g. $to_require = "file.php"; )
    - Fixed "Access Violation" when an "include", "include_once",
      "require" or "require_once" was at the end of a file
    - Fixed error when detecting the value of define()-constants
    - Removed "place holder" methods in TPHPEntity and TPHPConstant
      by making them "abstract"

  Version 2.2 (March 25 2005)
    - ParseEntityData() is not called anymore when setting SourceText
      property of TPHPSource because the call is unnecessary
    - Fixed error recognizing entities that were splitted up by php-tags
    - Fixed determining base class name in PHP4-Only mode
    - Fixed error that entites sometimes had incorrect "TextPos" properties
      (Position indicated by TextPos was 1 char before the real entity)

  Version 2.1 (March 04 2005)
    - Fixed error that functions that returned a reference (method name was
      beginning with "&") were not recognized
    - Added new booolean property "ReturnsReference" to TPHPMethod and all
      descendandts that indicates if a function does return a reference (or not)

  Version 2.0 (January 29 2005)
    - Fixed error that some IRegEx interfaces were not set to "nil" in the
      finalization section of the unit
    - Fixed error when recognizing default method parameters that contained
      a comma.
    - Fixed error that include files were not recognized when they were the
      last entity in the document and did not have a comma at the end.
      Note that the matched result may be too long in this case, because
      the RegEx alternatively matches until the end of the line when no comma
      was found.
    - Fixed bug that the parsing was continued when no end / begin tag
      was found. This bug only ocurred if there was a PHP tag in comments
      and the PHP Inspector tried to find the next correct tag.
    - Fixed error that define()-constants were not recognized if
      the constant value was a variable.
    - Fixed error that define()-constants were not recognized if
      the last parameter was a non-decimal value.
    - Fixed bug that the temporary file, that was created in the
      SyntaxCheck() function, was not deleted if the syntaxcheck
      has been successful (PHP.exe returned zero).
    - Fixed infinite loops in TPHPSource.FindInterface(),
      TPHPSource.FindClass() and TPHPSource.FindMethod().
    - Fixed declaration of TPHPSource.FindMethod() so that
      Interfaces can passed through the AClass parameter again.
    - Fixed error that the implemented interface of a class was not recognized
      if it additionally has extended another class.
    - Added option to collect local used variables by methods.
      This feature is disabled by default. Turn on the $FUNCTIONVARIABLES
      compilerswitch to enable the feature.
    - Added support for class constants in interfaces (ConstantCount and
      Constants[] properties)
    - Added property "ExtendsInterface" for TPHPInterface. It contains
      the name of the interface that is extended by the TPHPInterface.
    - Added supported for multiple inheritance that is provied by interfaces
    - RegEx constants are now delcared as resourcestrings.

  Version 1.9 (December 9 2004)
    - Fixed error in TPHPClassMethod and TPHPMethod
      BeginBracket and EndBracket calculation
    - Fixed error that caused that constant was only recognized when it was
      embedded in single quotes.
    - Fixed incorrectly displayed line breaks / spaces at the end of a
      include-file
    - Fixed bug in the PHP-Tag detection when a begin or end-tag was embedded
      in a string or in a comment
    - Fixed bug in the PHP-Tag detection when the StrictPHPTagCheck property
      was false and the source had no ending ?> (the source was not analyzed
      in that case)
    - Added class TPHPDefineConstant for constants defined per "define"
      (also supported in PHP4-mode)
    - Replaced [^$] in IncludeRegEx with an PCRE assertion (speed improvement)

  Version 1.8 (December 5 2004)
    - Added TPHPConstant class to support constans in PHP5
    - Enhanced TPHPIncludeFile class (IncludeKind, IncludeKindStr and
      IncludeFileName properties)
    - Renamed class TPHPClass to TPHPCustomClass and removed some properties
      TPHPClass is now derived from TPHPCustomClass and adds the needed
      properties as the other TPHPCustomClass descendants do.
    - Fixed errorneus indexes in entities which are within a class
    - Fixed bug when detecting php tags (operator was set to "greedy" and
      matched to many results)
    - Fixed bug when detecting include files (variables named $include etc.
      were matched)

  Version 1.7 (November 23 2004)
    - Fixed bug that caused TPHPInterface not recognizing methods any more
      when $PARSEMETHODBODY was specified. That results in the new
      TPHPInterfaceMethod class that does not need occurences with brackets.
      Furthermore it does not allow abstract, final or static methods.
    - Fixed errors in "php4-only" mode ($PHP4 defined) that caused wrong
      detection of the TPHPMethod.Name, TPHPClass.Name and the
      TPHPClass.ExtendsClass properties.
    - Added property "StrictPHPTagCheck" to TPHPSource to able to parse
      php files that have an opening <? but not an ending ?>
      (note: the missing ?> may cause problems with PHP/HTML mixed files)
    - Fixed TPHPVariable.VarName when the variable was initialised with
      a value
    - Added properties IsStatic for TPHPMethod and TPHPVariable (php5 mode only)
    - Added TPHPClassMethod for methods within a class. TPHPMethod
      hasn't got the properties IsAbtract, IsFinal and IsStatic anymore,
      because methods in a php file outside a class are not allowed to have
      those states. Visibility of TPHPMethod is public.
      These changes have a positive impact on the speed when determining
      methods that are not within a class in TPHPSource.ParseEntity()

  Version 1.6 (November 22 2004)
    - Added class TPHPIncludeFile that is used for includes instead
      of TPHPEntity
    - Made TPHPEntity as the ancestor of TPHPClass
    - Some speed and code optimizations
    - Only php code between <? ?> is parsed in TPHPSource
    - An interface is now checked if it is embedded in a comment before the
      whole interface body is read in (application of patch in v.1.4)
    - Replaced AnsiCompareStr() through explicit string comparisons (faster)
    - Added properties Body and Head to TPHPMethod
    - Added properties BeginBracket and EndBracket to TPHPMethod, TPHPClass and
      TPHPInterface
    - Added ability to parse the method body to be able to determine
      the beginning and the ending bracket of the method.
      You can disable this function if it is unnecessary.
      Use the $PARSEMETHODBODY compilerswitch to disable it.
    - Added Parent property to TPHPEntity class. Usually the Parent
      of Methods and Variables is the class they are in. If, for example,
      a method is not in a class its parent will be the TPHPSource instance.
      This is applied for all entites that are not in a class or in an
      interface.

  Version 1.5 (November 15 2004)
    - Now parsing comments blocks only when necessary. That optimizes speed
      when parsing documents with long comments blocks where no entites are
      located.
    - Optimized comment detection algorithm
    - The php.exe is now started with SW_HIDE param instead of the
      SW_MINIMIZE param
    - Pre-Initiating IRegEx Interfaces. Elseware the Regexes are created
      multiple times when parsing the source
    - Renamed class TPHPEntry in TPHPEntity

  Version 1.4 (November 14 2004):
    - Put "CheckIsCommented" Function in own class so that the source must
      not be searched multiple times for comments
    - Fixed EStackOverflow (caused crash of the application) when
      determining comments
    - A class is now checked if it is embedded in a comment before the
      whole class body is read in.

  Version 1.3 (November 10 2004):
    - Fixed error recognizing private and protected methods (PHP 5 only)
    - Fixed wrong entity position if the class declared like this:
      class x extends y
    - Added support for abstract classes and final methods (PHP 5 only)
    - Added support for interface implementor classes (PHP 5 only)
    - Added support for interfaces (PHP 5 only) (TPHPInterface class)
      Use Interfaces[] and InterfaceCount provided by TPHPSource
    - Added properties "ExtendsClass" and "IsAbstractClass" (PHP 5 only)
      "ImplementedInterface" (PHP 5 only) to TPHPClass class.
    - Added property "IsAbstract" and "IsFinal" (PHP 5 only) in
      TPHPMethod class
    - Added methods "FindClass", "FindInterface" (PHP 5 only) and "FindMethod to
      TPHPSource class

  Version 1.2 (November 7 2004):
    - Fixed wrong index of entities in classes
    - Fixed error when recognizing entities embedded in doubles or
      single quotes. ("CheckIsCommented" function)
    - Fixed error that made the inspector recognize commented source as
      real entitites. ("CheckIsCommented" function)
    - Fixed error that fields were not cleared if the ParseSource() function
      was called.
    - Added ability to make a syntax check for the sourcetext
      (PHP installation on local harddrive required; version 4.3.0 or higher)

  Version 1.1 (November 2 2004):
   - Fixed error in "CheckIsCommented" function

  Version 1.0 (November 1 2004):
   - Initial Release
}

interface

uses
  Windows, PCRE, Contnrs, Classes;

// Compiler Switches
// ----------------------------------------------------------------------------

// if defined, PHP 5 support is dropped (not defined by default).
{.$DEFINE PHP4}

// if defined, PHP methods will be parsed completely including the methodbody.
// This takes a bit more time than just parsing the method head, but it
// is recommended to enable this compiler-switch (defined by default)
{$DEFINE PARSEMETHODBODY}

// if defined, locally used variables of methods (of any kind) will be
// collected. Maybe this takes more time than it may be useful, so,
// by default, it is undefined.
// $PARSEMETHODBODY must also be turned on for this feature to work.
{.$DEFINE FUNCTIONVARIABLES}

// if defined, entities with the same name will not be read in more than
// one time. this only affects classes, interfaces and methods.
// this option is not defined by default.
{.$DEFINE NODUPLICATES}

// ----------------------------------------------------------------------------

// current version
resourcestring
  PHPINSPECTORVERSION = '2.5';

type
  TPHPVarVisibility = (vvPublic{$IFNDEF PHP4}, vvPrivate, vvProtected{$ENDIF});
  TPHPIncludeKind   = (ikInclude, ikIncludeOnce, ikRequire, ikRequireOnce);

type
  TPHPCustomClass = class;

  TPHPEntity = class(TObject)
  protected
    FParent: TPHPCustomClass;
    FOccurence: string;
    FTextPos: Integer;
    procedure ParseEntity; overload; virtual; abstract;
    procedure ParseEntity(const Match: IMatch); overload; virtual; abstract;
  public
    property Parent: TPHPCustomClass read FParent;
    property Occurence: string read FOccurence;
    property TextPos: Integer read FTextPos;
    constructor Create(Parent: TPHPCustomClass;
      const Occurence: string; TextPos: Integer); overload; virtual;
  end;

  TPHPMethod = class(TPHPEntity)
  protected
    FMethodName: string;
    FMethodHead: string;
    {$IFDEF PARSEMETHODBODY}
    FMethodBody: string;
    FBeginBracket: Integer;
    FEndBracket  : Integer;
    {$ENDIF}
    FParams: TStringList;
    {$IFDEF FUNCTIONVARIABLES}
    FLocalVars: TStringList;
    {$ENDIF}
    FReturnsReference: Boolean;
    FVisibility: TPHPVarVisibility;

    procedure CreateInitFields;
    constructor Create(Parent: TPHPCustomClass;
      const Occurence: string; TextPos: Integer; const NameMatch,
      BodyMatch: IMatch); overload;
    function GetParamCount: Integer;
    function GetParameters(Index: Integer): string;
    {$IFDEF FUNCTIONVARIABLES}
    function GetLocalVariablesCount: Integer;
    function GetLocalVariables(Index: Integer): string;
    {$ENDIF}
  protected
    {$IFDEF FUNCTIONVARIABLES}
    procedure ParseLocalVariables; virtual;
    {$ENDIF}
    procedure ParseParameters(const Parameters: string); virtual;
    procedure ParseEntity; overload; override;
    procedure ParseEntity(const NameMatch, BodyMatch: IMatch); overload;
      virtual;
  public
    property Name: string read FMethodName;
    property Head: string read FMethodHead;
    {$IFDEF PARSEMETHODBODY}
    property Body: string read FMethodBody;
    property BeginBracket: Integer read FBeginBracket;
    property EndBracket: Integer read FEndBracket;
    {$ENDIF}
    property ParamCount: Integer read GetParamCount;
    property Parameters[Index: Integer]: string read GetParameters;
    {$IFDEF FUNCTIONVARIABLES}
    property LocalVariables[Index: Integer]: string
      read GetLocalVariables;
    property LocalVariablesCount: Integer read GetLocalVariablesCount;
    {$ENDIF}
    property ReturnsReference: Boolean read FReturnsReference;
    property Visibility: TPHPVarVisibility read FVisibility;

    constructor Create(Parent: TPHPCustomClass;
      const Occurence: string; TextPos: Integer); overload; override;
    destructor Destroy; override;
  end;

  TPHPClassMethod = class(TPHPMethod)
  private
    {$IFNDEF PHP4}
    FIsAbstract: Boolean;
    FIsFinal: Boolean;
    FIsStatic: Boolean;
    {$ENDIF}
  protected
    procedure ParseEntity; overload; override;
    procedure ParseEntity(const NameMatch, BodyMatch: IMatch); overload;
      override;
  public
    {$IFNDEF PHP4}
    property IsAbstract: Boolean read FIsAbstract;
    property IsFinal: Boolean read FIsFinal;
    property IsStatic: Boolean read FIsStatic;
    {$ENDIF}
  end;

  {$IFNDEF PHP4}
  TPHPInterfaceMethod = class(TPHPMethod)
  protected
    procedure ParseEntity(const NameMatch, BodyMatch: IMatch); override;
  end;
  {$ENDIF}

  TPHPVariable = class(TPHPEntity)
  private
    FVarName: string;
    FInitiatedValue: string;
    FVisibility: TPHPVarVisibility;
    {$IFNDEF PHP4}
    FIsStatic: Boolean;
    {$ENDIF}
    // function  VisibilityToString(Visibility: TPHPVarVisibility): String;
    constructor Create(Parent: TPHPCustomClass; const Occurence: string;
      TextPos: Integer; const Match: IMatch); overload;
  protected
    procedure ParseEntity; overload; override;
    procedure ParseEntity(const Match: IMatch); overload; override;
  public
    property VarName: string read FVarName;
    property InitiatedValue: string read FInitiatedValue;
    {$IFNDEF PHP4}
    property IsStatic: Boolean read FIsStatic;
    {$ENDIF}
    property Visibility: TPHPVarVisibility read FVisibility;

    constructor Create(Parent: TPHPCustomClass; const Occurence: string;
      TextPos: Integer); overload; override;
  end;

  TPHPConstant = class(TPHPEntity)
  protected
    FName: string;
    FValue: string;
    constructor Create(Parent: TPHPCustomClass; const Occurence: string;
      TextPos: Integer; const Match: IMatch); overload;
    procedure ParseEntity; reintroduce; overload; virtual; abstract;
    procedure ParseEntity(const Match: IMatch); reintroduce; overload; virtual;
      abstract;
  public
    property Name: string read FName;
    property Value: string read FValue;
    constructor Create(Parent: TPHPCustomClass; const Occurence: string;
      TextPos: Integer); overload; override;
  end;

  {$IFNDEF PHP4}
  TPHPClassConstant = class(TPHPConstant)
  protected
    procedure ParseEntity; overload; override;
    procedure ParseEntity(const Match: IMatch); overload; override;
  end;
  {$ENDIF}

  TPHPDefineConstant = class(TPHPConstant)
  protected
    procedure ParseEntity; overload; override;
    procedure ParseEntity(const Match: IMatch); overload; override;
  end;

  TPHPIncludeFile = class(TPHPEntity)
  private
    FIncludeFileName: string;
    FIncludeKind: TPHPIncludeKind;
    FIncludeKindStr: string;
    constructor Create(Parent: TPHPCustomClass; const Occurence: string;
      TextPos: Integer; const Match: IMatch); overload;
  protected
    procedure ParseEntity; overload; override;
    procedure ParseEntity(const Match: IMatch); overload; override;
  public
    property IncludeFileName: string read FIncludeFileName;
    property IncludeKind: TPHPIncludeKind read FIncludeKind;
    property IncludeKindStr: string read FIncludeKindStr;
    constructor Create(Parent: TPHPCustomClass; const Occurence: string;
      TextPos: Integer); overload; override;
  end;

  TPHPComment = class(TObject)
  private
    FIndex, FLength: Integer;
  public
    property Index: Integer read FIndex;
    property Length: Integer read FLength;
    constructor Create(Index, Length: Integer);
  end;

  TPHPComments = class(TObject)
  private
    FSource: string;
    FCacheEntryIndex: Integer;
    FComments: TObjectList;
    procedure SetSource(const Value: string);
    function GetComments(Index: Integer): TPHPComment;
    function GetCommentCount: Integer;
    procedure ParseTo(AIndex: Integer);
  public
    property Source: string read FSource write SetSource;
    property Comments[Index: integer]: TPHPComment read GetComments;
    property CommentCount: Integer read GetCommentCount;

    constructor Create;
    destructor Destroy; override;
    procedure FullParse;
    function CheckIsCommented(AStartPos, ALength: Integer): Boolean;
  end;

  TPHPCustomClass = class(TPHPEntity)
  protected
    FComments: TPHPComments;
    FName: string;
    {$IFNDEF PHP4}
    FIsAbstract: Boolean;
    FImplements: string;
    FConstants: TObjectList;    
    {$ENDIF}
    FMethods: TObjectList;
    FVariables: TObjectList;
    FExtends: string;
    FBeginBracket: Integer;
    FEndBracket: Integer;
    function GetMethodCount: Integer;
    function GetMethods(Index: Integer): TPHPClassMethod;
    function GetMethodsEx(Index: Integer): TPHPMethod; virtual;
    function GetVariableCount: Integer;
    function GetVariables(Index: Integer): TPHPVariable;
    {$IFNDEF PHP4}
    function GetConstantCount: Integer; virtual;
    function GetConstants(Index: Integer): TPHPClassConstant;
    {$ENDIF}
    procedure SetSourceText(const Value: string); virtual;
    function DetermineEntityData(const NameMatch, BodyMatch: IMatch): string;
      virtual;
    function DetermineEntityName(const Text: string): string; virtual;
    procedure ParseEntity; overload; override;
    procedure ParseEntity(const Match: IMatch); overload; override;
    constructor Create(Parent: TPHPCustomClass; const ClassNameMatch,
      ClassBodyMatch: IMatch; const SourceText: string; TextPos: Integer);
      overload; virtual;
  public
    property Name        : string read FName;
    property Occurence   : string read FOccurence write SetSourceText;
    property BeginBracket: Integer read FBeginBracket;
    property EndBracket  : Integer read FEndBracket;

    constructor Create(Parent: TPHPCustomClass; const Occurence: string;
      TextPos: Integer); overload; override;
    destructor Destroy; override;
    function FindEntityAt(APos: Integer): TPHPEntity; virtual;
  end;

  TPHPClass = class(TPHPCustomClass)
  public
    property MethodCount: Integer read GetMethodCount;
    property Methods[Index: Integer]: TPHPClassMethod read GetMethods;
    property VariableCount: Integer read GetVariableCount;
    property Variables[Index: Integer]: TPHPVariable read GetVariables;
    {$IFNDEF PHP4}
    property ConstantCount: Integer read GetConstantCount;
    property Constants[Index: Integer]: TPHPClassConstant read GetConstants;
    property IsAbstractClass: Boolean read FIsAbstract;
    property ImplementedInterface: string read FImplements;
    {$ENDIF}
    property ExtendsClass: string read FExtends;        
  end;

  {$IFNDEF PHP4}
  TPHPInterface = class(TPHPCustomClass)
  protected
    constructor Create(Parent: TPHPCustomClass; const ClassNameMatch,
      ClassBodyMatch: IMatch; const SourceText: string; TextPos: Integer);
      overload; override;
    function GetInterfaceMethod(Index: Integer): TPHPInterfaceMethod;
    function DetermineEntityName(const Text: string): string; override;
    function DetermineEntityData(const NameMatch, BodyMatch: IMatch): string;
      override;
    procedure ParseEntity; override;
  public
    property MethodCount: Integer read GetMethodCount;
    property Methods[Index: Integer]: TPHPInterfaceMethod read
      GetInterfaceMethod;
    property ConstantCount: Integer read GetConstantCount;
    property Constants[Index: Integer]: TPHPClassConstant read GetConstants;
    property ExtendsInterface: string read FExtends;
    constructor Create(Parent: TPHPCustomClass;
      const SourceText: string; TextPos: Integer); override;
  end;
  {$ENDIF}

  TPHPSource = class(TPHPClass)     
  private
    FClasses: TObjectList;
    {$IFNDEF PHP4}
    FInterfaces: TObjectList;
    {$ENDIF}
    FIncludes: TObjectList;
    FPHPExe: string;
    FSyntaxErrors: TStrings;
    FResultOutPut: TStrings;
    FStrictPHPTagCheck: Boolean;
    {$IFDEF PHP4}
    FConstants: TObjectList;
    function GetConstantCount: Integer;
    {$ENDIF}
    function GetConstantEx(Index: Integer): TPHPDefineConstant;    
    function GetClassCount: Integer;
    function GetClasses(Index: Integer): TPHPClass;
    {$IFNDEF PHP4}
    function GetInterfaceCount: Integer;
    function GetInterfaces(Index: Integer): TPHPInterface;
    {$ENDIF}
    function GetIncludeCount: Integer;
    function GetIncludes(Index: Integer): TPHPIncludeFile;
  protected
    function GetMethodsEx(Index: Integer): TPHPMethod; override;
    procedure SetSourceText(const Value: string); override;
    procedure ParseEntity; override;
  public
    property SourceText: string read FOccurence write SetSourceText;
    property StrictPHPTagCheck: Boolean read FStrictPHPTagCheck
      write FStrictPHPTagCheck;
    property PHPExe: string read FPHPExe write FPHPExe;
    property SxntaxErrors: TStrings read FSyntaxErrors;
    property OutPut: TStrings read FResultOutPut;
    property ClassCount: Integer read GetClassCount;
    property Classes[Index: Integer]: TPHPClass read GetClasses;
    property ConstantCount: Integer read GetConstantCount;
    property Constants[Index: Integer]: TPHPDefineConstant read GetConstantEx;
    {$IFNDEF PHP4}
    property InterfaceCount: Integer read GetInterfaceCount;
    property Interfaces[Index: Integer]: TPHPInterface read GetInterfaces;
    {$ENDIF}
    property IncludeCount: Integer read GetIncludeCount;
    property Includes[Index: Integer]: TPHPIncludeFile read GetIncludes;
    property Methods[Index: Integer]: TPHPMethod read GetMethodsEx;

    constructor Create(const SourceText: string); overload;
    constructor Create(Parent: TPHPCustomClass; const Occurence: string;
      TextPos: Integer); overload; override;
    destructor Destroy; override;
    class function Version: string;
    procedure LoadFromFile(const FileName: string);
    function SyntaxCheck(CheckForRuntimeErrors: Boolean = False;
      const FileName: string = ''): Boolean;
    function FindClass(const ClassName: string): Integer;
    {$IFNDEF PHP4}
    function FindInterface(const InterfaceName: string): Integer;
    {$ENDIF}
    function FindMethod(AClass: TPHPCustomClass;
      const MethodName: string): TPHPMethod;
    function FindEntityAt(APos: Integer): TPHPEntity; override;
  end;

implementation

uses SysUtils, Math;

// constants for RegExes
resourcestring
  {$IFDEF PHP4}
  ABSTRACTCLASS = '';
  CLASSSTR      = '((\s+?)extends(\s+?)(\w+?)|)';
  VARSEARCHSTR  = 'var';
  {$ELSE}
  ABSTRACTCLASS = '(((abstract)(\s+?))|)';
  CLASSSTR      = '((((\s+)(extends)(\s+)(\w+?)|)(((\s+)(implements)(\s+)' +
    '(.*)(\s*)(?=\{))|))|)';
  VARSEARCHSTR  = '(public|var|private|protected)';
  {$ENDIF}
  VARNAMESTR    = '\$(\w+?)(((\s*)(\=(\s*)(.*))|));';

// used RegExes
var
  VarRegEx, DefaultMethodRegEx, MethodRegEx, ClassNameRegEx, ClassBodyRegEx,
  IncludeRegEx, StrictSourceRegEx, SourceRegEx, BeginSourceRegEx,
  EndSourceRegEx, CommentRegEx, DefineConstantRegEx, MethodParamRegEx: IRegEx;
  {$IFDEF NODUPLICATES}
     SimpleClassNameRegEx: IRegEx;
     {$IFNDEF PHP4}SimpleInterfaceNameRegEx: IRegEx;{$ENDIF}
  {$ENDIF}
  {$IFDEF FUNCTIONVARIABLES}VariableRegEx: IRegEx;{$ENDIF}
  {$IFNDEF PHP4}
  ClassMethodRegEx, InterfaceNameRegEx, ConstantRegEx: IRegEx;
  {$ENDIF}

function CreateTempFile: string;

  function GetTempDir: string;
  var
    Buf: array[0..MAX_PATH] of Char;
  begin
    GetTempPath(SizeOf(Buf), Buf);
    Result := Buf;
  end;

var
  Buf: array[0..MAX_PATH] of Char;
begin
  GetTempFileName(PChar(GetTempDir), 'tmp', 0, Buf);
  Result := Buf;
end;

// based on http://www.swissdelphicenter.com/de/showcode.php?id=990
function RunCaptured(const DirName, ExeName, CmdLine: string;
  OutPut: TStrings; var ReturnCode: Cardinal): Boolean;
var
  StartupInfo: TStartupInfo;
  ProcInfo: TProcessInformation;
  FileName: string;
  FileHandle: THandle;
  SecurityAttributes: TSecurityAttributes;
begin
  Result := False;
  try
    // set a temporary file
    FileName := 'Test.tmp';
    FillChar(SecurityAttributes, SizeOf(SecurityAttributes), #0);
    SecurityAttributes.nLength := SizeOf(SecurityAttributes);
    SecurityAttributes.bInheritHandle := True;
    FileHandle := Windows.CreateFile(PChar(FileName),
      GENERIC_WRITE, FILE_SHARE_WRITE, @SecurityAttributes, CREATE_ALWAYS,
      FILE_ATTRIBUTE_NORMAL, 0);
    try
      FillChar(StartupInfo, SizeOf(StartupInfo), #0);
      StartupInfo.cb          := SizeOf(StartupInfo);
      StartupInfo.hStdOutput  := FileHandle;
      StartupInfo.dwFlags     := STARTF_USESTDHANDLES or STARTF_USESHOWWINDOW;
      StartupInfo.wShowWindow := SW_HIDE;
      // start the program 
      if CreateProcess(nil, PChar(ExeName + ' ' + CmdLine), nil, nil, True,
        0, nil, PChar(DirName), StartupInfo, ProcInfo) then
      begin
        SetPriorityClass(ProcInfo.hProcess, IDLE_PRIORITY_CLASS);
        WaitForSingleObject(ProcInfo.hProcess, Infinite);
        GetExitCodeProcess(ProcInfo.hProcess, ReturnCode);
        // PHP.exe returns 255 on parse errors!
        Result := True;
        CloseHandle(ProcInfo.hThread);
        CloseHandle(ProcInfo.hProcess);
        Windows.CloseHandle(FileHandle);
        // Add the output
        if (OutPut <> nil) then
          OutPut.LoadFromFile(FileName);
      end;
      Windows.DeleteFile(PChar(FileName));
    except
      Windows.CloseHandle(FileHandle);
      Windows.DeleteFile(PChar(FileName));
      Result := False;
    end;
  finally
  end;
end;

function BuildMethodVisibility(
  {$IFNDEF PHP4}AbstractAndFinalMethods: Boolean = True{$ENDIF}): string;
const
  {$IFDEF PHP4}
  VISIBILITY : array[1..1] of string = ('');
  {$ELSE}
  VISIBILITY : array[1..3] of string = ('public', 'private', 'protected');
resourcestring
  ABSTRACTANDFINALMETHOD  = '((abstract|final)(\s+?)|)';
  {$ENDIF}
{$IFNDEF PHP4}
var
  i: integer;
  s: string;
begin
  s := '';
  for i := low(VISIBILITY) to high(VISIBILITY) do
    s := s + VISIBILITY[i] + '|';
  Delete(s, Length(s), 1);
  Result := '(((' + S + ')(\s+?))|)';
  if AbstractAndFinalMethods then
    Result := ABSTRACTANDFINALMETHOD + Result;
  {$ELSE}
begin
  Result := '';
  {$ENDIF}
end;

function BuildStatic: string;
begin
  {$IFDEF PHP4}
  Result := '';
  {$ELSE}
  Result := '(static(\s+?)|)';
  {$ENDIF}
end;

function StringToVisibility(const Str: string): TPHPVarVisibility;
begin
  {$IFDEF PHP4}
  Result := vvPublic;
  {$ELSE}
  if Str = 'public' then
    Result := vvPublic else
  if Str = 'private' then
    Result := vvPrivate else
  if Str = 'protected' then
    Result := vvProtected else
  Result := vvPublic;
  {$ENDIF}
end;

{ TPHPEntity }

constructor TPHPEntity.Create(Parent: TPHPCustomClass; const Occurence: string;
  TextPos: Integer);
begin
  inherited Create;
  FParent := Parent;
  FOccurence := Occurence;
  FTextPos := TextPos;
end;

{ TPHPMethod }

constructor TPHPMethod.Create(Parent: TPHPCustomClass; const Occurence: string;
  TextPos: Integer);
begin
  inherited;
  CreateInitFields;
  ParseEntity;
end;

constructor TPHPMethod.Create(Parent: TPHPCustomClass; const Occurence: string;
  TextPos: Integer; const NameMatch, BodyMatch: IMatch);
begin
  inherited Create(Parent, Occurence, TextPos);
  CreateInitFields;
  ParseEntity(NameMatch, BodyMatch);
end;

destructor TPHPMethod.Destroy;
begin
  FParams.Free;
  {$IFDEF FUNCTIONVARIABLES}
  FLocalVars.Free;
  {$ENDIF}
  inherited;
end;

procedure TPHPMethod.CreateInitFields;
begin
  {$IFDEF FUNCTIONVARIABLES}
  FLocalVars := TStringList.Create;
  FLocalVars.Sorted := True;
  FLocalVars.Duplicates := dupIgnore;
  {$ENDIF}
  FParams := TStringList.Create;
end;

function TPHPMethod.GetParamCount: Integer;
begin
  Result := FParams.Count;
end;

function TPHPMethod.GetParameters(Index: Integer): string;
begin
  Result := FParams[Index];
end;

{$IFDEF FUNCTIONVARIABLES}
function TPHPMethod.GetLocalVariables(Index: Integer): string;
begin
  Result := FLocalVars[Index];
end;

function TPHPMethod.GetLocalVariablesCount: Integer;
begin
  Result := FLocalVars.Count;
end;

procedure TPHPMethod.ParseLocalVariables;
var
  MatchColl: IMatchCollection;
  i: Integer;
begin
  FLocalVars.Clear;
  MatchColl := VariableRegEx.Matches(FOccurence);
  for i := 0 to MatchColl.Count-1 do
    FLocalVars.Add(MatchColl.Items[i].Value);
end;
{$ENDIF}

procedure TPHPMethod.ParseParameters(const Parameters: string);
var
  i: Integer;
  ParamMatch: IMatchCollection;
begin
  FParams.Clear;
  ParamMatch := MethodParamRegEx.Matches(Parameters);
  for i := 0 to ParamMatch.Count-1 do
    FParams.Add(ParamMatch.Items[i].Value);
end;

procedure TPHPMethod.ParseEntity(const NameMatch, BodyMatch: IMatch);
begin
  if (NameMatch <> nil) and (NameMatch.Matched) then
  with NameMatch do
  begin
    FVisibility := vvPublic;
    FReturnsReference := (Groups.Items[2].Value = '&');
    FMethodName := Groups.Items[3].Value;
    ParseParameters(Groups.Items[5].Value);
    FMethodHead := NameMatch.Value;
  end;
  {$IFDEF PARSEMETHODBODY}
  if (BodyMatch <> nil) and (BodyMatch.Matched) then
  with BodyMatch do
  begin
    // fix v1.9: need to subtract NameMatch.Index from the TextPos
    // because the constructor is called with NameMatch.Index added to
    // the text position and the BodyMatch.Index is not relative to
    // the NameMatch.Index
    FBeginBracket := (FTextPos - NameMatch.Index) + BodyMatch.Index;
    FEndBracket  :=  (FTextPos - NameMatch.Index) + BodyMatch.Index +
      BodyMatch.Length;
    FMethodBody  := BodyMatch.Value;
    {$IFDEF FUNCTIONVARIABLES}
    ParseLocalVariables;
    {$ENDIF}
  end;
  {$ENDIF}
end;

procedure TPHPMethod.ParseEntity;
var
  NameMatch, BodyMatch: IMatch;
begin
  with DefaultMethodRegEx do
    NameMatch := Match(FOccurence);
  {$IFDEF PARSEMETHODBODY}
  with ClassBodyRegEx do
    BodyMatch := Match(FOccurence);
  {$ELSE}
  BodyMatch := nil;
  {$ENDIF}
  ParseEntity(NameMatch, BodyMatch);
end;

{ TPHPClassMethod }
procedure TPHPClassMethod.ParseEntity;
var
  NameMatch, BodyMatch: IMatch;
begin
  with MethodRegEx do
    NameMatch := Match(FOccurence);
  {$IFDEF PARSEMETHODBODY}
  with ClassBodyRegEx do
    BodyMatch := Match(FOccurence);
  {$ELSE}
  BodyMatch := nil;
  {$ENDIF}
  ParseEntity(NameMatch, BodyMatch);
end;

procedure TPHPClassMethod.ParseEntity(const NameMatch, BodyMatch: IMatch);
var
  Params: string;
begin
  if (NameMatch <> nil) then
  begin
    if NameMatch.Matched then
    with NameMatch do
    begin
      // fix v1.7
      {$IFNDEF PHP4}
      FIsAbstract := Trim(Groups.Items[1].Value) = 'abstract';
      FIsFinal := Trim(Groups.Items[1].Value) = 'final';
      FVisibility := StringToVisibility(Groups.Items[6].Value);
      FIsStatic := Trim(Groups.Items[8].Value) = 'static';
      FReturnsReference := (Groups.Items[11].Value = '&');      
      FMethodName := Groups.Items[12].Value;
      Params := Groups.Items[14].Value;
      {$ELSE}
      FVisibility :=
        StringToVisibility(Trim(Groups.Items[1].Value));
      FMethodName := Groups.Items[4].Value;
      Params := Groups.Items[6].Value;
      {$ENDIF}
      ParseParameters(Params);
      FMethodHead := NameMatch.Value; 
    end;
    {$IFDEF PARSEMETHODBODY}
    if (BodyMatch <> nil) and (BodyMatch.Matched) then
    with BodyMatch do
    begin
      // fix v1.9: need to subtract NameMatch.Index from the TextPos
      // because the constructor is called with NameMatch.Index added to
      // the text position and the BodyMatch.Index is not relative to
      // the NameMatch.Index 
      FBeginBracket := (FTextPos - NameMatch.Index) + BodyMatch.Index;
      FEndBracket  :=  (FTextPos - NameMatch.Index) + BodyMatch.Index +
        BodyMatch.Length;
      FMethodBody  := BodyMatch.Value;
      {$IFDEF FUNCTIONVARIABLES}
      ParseLocalVariables;
      {$ENDIF}
    end;
    {$ENDIF}
  end;  
end;

{ TPHPInterfaceMethod }

{$IFNDEF PHP4}
procedure TPHPInterfaceMethod.ParseEntity(const NameMatch,
  BodyMatch: IMatch);
begin
  // Bodymatch is ignored!
  if (NameMatch <> nil) and (NameMatch.Matched) then
  with NameMatch do
  begin
    FVisibility :=
      StringToVisibility(Groups.Items[3].Value);
    FReturnsReference := (Groups.Items[6].Value = '&');      
    FMethodName := Groups.Items[7].Value;
    ParseParameters(Groups.Items[9].Value);
    FMethodHead := NameMatch.Value;
    {$IFDEF PARSEMETHODBODY}
    FEndBracket := -1;
    FBeginBracket := -1;
    FMethodBody := '';
    {$ENDIF}
  end;
end;
{$ENDIF}

{ TPHPVariable }

constructor TPHPVariable.Create(Parent: TPHPCustomClass;
  const Occurence: string; TextPos: Integer);
begin
  inherited;
  ParseEntity;
end;

constructor TPHPVariable.Create(Parent: TPHPCustomClass;
  const Occurence: string; TextPos: Integer; const Match: IMatch);
begin
  inherited Create(Parent, Occurence, TextPos);
  ParseEntity(Match);
end;

(*
function TPHPVariable.VisibilityToString(
  Visibility: TPHPVarVisibility): String;
{$IFNDEF PHP4}
const
  STR : array[TPHPVarVisibility] of String = ('public', 'private',
    'protected');
{$ENDIF}
begin
  {$IFDEF PHP4}
  Result := 'var';
  {$ELSE}
  Result := STR[Visibility];
  {$ENDIF}
end;
*)

procedure TPHPVariable.ParseEntity;
var
  MatchColl: IMatch;
begin
  MatchColl := VarRegEx.Match(FOccurence);
  if MatchColl.Matched then
    ParseEntity(MatchColl);
end;

procedure TPHPVariable.ParseEntity(const Match: IMatch);
begin
  if (Match <> nil) and (Match.Matched) then
  with Match do
  begin
    FVisibility := StringToVisibility(Groups.Items[1].Value);
    {$IFNDEF PHP4}
    FIsStatic := Trim(Groups.Items[3].Value) = 'static';
    FVarName := Groups.Items[5].Value;
    FInitiatedValue := Trim(Groups.Items[11].Value);
    {$ELSE}
    FVarName := Groups.Items[2].Value;
    FInitiatedValue := Trim(Groups.Items[8].Value);    
    {$ENDIF}
  end;
end;

{ TPHPConstant }

constructor TPHPConstant.Create(Parent: TPHPCustomClass;
  const Occurence: string; TextPos: Integer; const Match: IMatch);
begin
  inherited Create(Parent, Occurence, TextPos);
  ParseEntity(Match);
end;

constructor TPHPConstant.Create(Parent: TPHPCustomClass;
  const Occurence: string; TextPos: Integer);
begin
  inherited;
  ParseEntity;
end;

{ TPHPClassConstant }
{$IFNDEF PHP4}
procedure TPHPClassConstant.ParseEntity;
var
  MatchColl: IMatch;
begin
  with ConstantRegEx do
    MatchColl := Match(FOccurence);
  ParseEntity(MatchColl);
end;

procedure TPHPClassConstant.ParseEntity(const Match: IMatch);
begin
  if (Match <> nil) and (Match.Matched) then
  begin
    FName := Match.Groups.Items[2].Value;
    FValue := Match.Groups.Items[5].Value;
  end;
end;
{$ENDIF}

{ TPHPDefineConstant }

procedure TPHPDefineConstant.ParseEntity;
var
  MatchColl: IMatch;
begin
  with DefineConstantRegEx do
    MatchColl := Match(FOccurence);
  ParseEntity(MatchColl);
end;

procedure TPHPDefineConstant.ParseEntity(const Match: IMatch);
begin
  if (Match <> nil) and (Match.Matched) then
  begin
    FName := Match.Groups.Items[2].Value;
    FValue := Match.Groups.Items[5].Value;
  end;
end;

{ TPHPIncludeFile }

constructor TPHPIncludeFile.Create(Parent: TPHPCustomClass;
  const Occurence: string; TextPos: Integer; const Match: IMatch);
begin
  inherited Create(Parent, Occurence, TextPos);
  ParseEntity(Match);
end;

constructor TPHPIncludeFile.Create(Parent: TPHPCustomClass;
  const Occurence: string; TextPos: Integer);
begin
  inherited;
  ParseEntity;
end;

procedure TPHPIncludeFile.ParseEntity;
var
  IncludeMatch: IMatch;
begin
  with IncludeRegEx do
    IncludeMatch := Match(FOccurence);
  ParseEntity(IncludeMatch);
end;

procedure TPHPIncludeFile.ParseEntity(const Match: IMatch);
var
  IncName: string;
begin
  if (Match <> nil) and (Match.Matched) then
  begin
    IncName := Match.Groups.Items[1].Value;
    if IncName = 'include' then
      FIncludeKind := ikInclude else
    if IncName = 'include_once' then
      FIncludeKind := ikIncludeOnce else
    if IncName = 'require' then
      FIncludeKind := ikRequire else
    if IncName = 'require_once' then
      FIncludeKind := ikRequireOnce else
    FIncludeKind := ikInclude;
    FIncludeKindStr := IncName;
    FIncludeFileName := Match.Groups.Items[3].Value;
    // fix v2.0
    // terminate char #13 at the end of the string (if present)
    // fix v2.3
    // Check if the string is longer than 0 chars
    if (Length(FIncludeFileName) > 0) and
      (FIncludeFileName[Length(FIncludeFileName)] = #13) then
      Delete(FIncludeFileName, Length(FIncludeFileName), 1);
  end;
end;

{ TPHPComment }

constructor TPHPComment.Create(Index, Length: Integer);
begin
  FIndex := Index;
  FLength := Length;
end;

{ TPHPComments }

constructor TPHPComments.Create;
begin
  inherited;
  FComments := TObjectList.Create;
end;

destructor TPHPComments.Destroy;
begin
  FComments.Free;
  inherited;
end;

function TPHPComments.GetComments(Index: Integer): TPHPComment;
begin
  Result := TPHPComment(FComments[Index]);
end;

function TPHPComments.GetCommentCount: Integer;
begin
  Result := FComments.Count;
end;

procedure TPHPComments.ParseTo(AIndex: Integer);
var
  MatchColl: IMatch;
begin
  if AIndex < FCacheEntryIndex then Exit;
  with CommentRegEx do
  begin
    MatchColl := Match(FSource, FCacheEntryIndex);
    while (MatchColl.Matched) and (MatchColl.Index <= AIndex) do
    begin
      FComments.Add(TPHPComment.Create(MatchColl.Index, MatchColl.Length));
      FCacheEntryIndex := MatchColl.Index;
      MatchColl := Match(FSource, Max(MatchColl.Index + MatchColl.Length,
        FCacheEntryIndex + 1));
    end;
  end;
end;

procedure TPHPComments.SetSource(const Value: string);
begin
  if (Value <> FSource) then
  begin
    FCacheEntryIndex := 0;
    FComments.Clear;
    FSource := Value;
  end;
end;

function TPHPComments.CheckIsCommented(AStartPos,
  ALength: Integer): Boolean;
var
  i, LastPos: integer;
  Comment: TPHPComment;
begin
  i := 0;
  Result := False;
  if FCacheEntryIndex <= AStartPos then
    ParseTo(AStartPos);
  while (i < FComments.Count) and (Result = False) do
  begin
    Comment := GetComments(i);
    LastPos := Comment.Index + Comment.Length;
    Result := (Comment.Index < AStartPos) and
      (LastPos > AStartPos + ALength);
    inc(i);
  end;
end;

procedure TPHPComments.FullParse;
begin
  ParseTo(Length(FSource));
end;

{ TPHPClass }

constructor TPHPCustomClass.Create(Parent: TPHPCustomClass;
  const Occurence: string;
  TextPos: Integer);
begin
  inherited Create(Parent, Occurence, TextPos);
  FMethods := TObjectList.Create;
  FVariables := TObjectList.Create;
  FComments := TPHPComments.Create;
  FExtends := '';
  {$IFNDEF PHP4}
  FConstants := TObjectList.Create;
  FIsAbstract := False;
  FImplements := '';
  {$ENDIF}
  FName := DetermineEntityName(Occurence);
  FTextPos := TextPos;
  ParseEntity;
end;

constructor TPHPCustomClass.Create(Parent: TPHPCustomClass;
  const ClassNameMatch, ClassBodyMatch: IMatch; const SourceText: string;
  TextPos: Integer);
begin
  // private constructor to parse information from already made matches!
  Create(Parent, '', TextPos);
  FName := DetermineEntityData(ClassNameMatch, ClassBodyMatch);
  FOccurence := SourceText;
  ParseEntity;
end;

destructor TPHPCustomClass.Destroy;
begin
  FMethods.Free;
  FVariables.Free;
  FComments.Free;
  {$IFNDEF PHP4}
  FConstants.Free;
  {$ENDIF}
  inherited;
end;

function TPHPCustomClass.DetermineEntityData(const NameMatch,
  BodyMatch: IMatch): string;
begin
  if (NameMatch <> nil) and (BodyMatch <> nil) then
  begin
    if NameMatch.Matched then
    begin
      {$IFNDEF PHP4}
      FIsAbstract := NameMatch.Groups.Items[3].Value = 'abstract';
      // fix v2.0
      // it is possible to extend a class and implement an interface, too.
      if NameMatch.Groups.Items[11].Value = 'extends' then
        FExtends := NameMatch.Groups.Items[13].Value else
      FExtends := '';
      if NameMatch.Groups.Items[17].Value = 'implements' then
        FImplements := NameMatch.Groups.Items[19].Value else
      FImplements := '';
      Result := NameMatch.Groups.Items[6].Value;
      {$ELSE}
      FExtends := NameMatch.Groups.Items[6].Value;
      Result := NameMatch.Groups.Items[2].Value;
      {$ENDIF}
    end else
      Result := '';
    if BodyMatch.Matched then
    begin
      FBeginBracket := FTextPos + BodyMatch.Index;
      FEndBracket   := FTextPos + BodyMatch.Index + BodyMatch.Length;
    end;
  end;
end;

function TPHPCustomClass.DetermineEntityName(const Text: string): string;
var
  ClassNameColl, ClassBodyColl: IMatch;
begin
  with ClassNameRegEx do
    ClassNameColl := Match(Text);
  with ClassBodyRegEx do
    ClassBodyColl := Match(Text);
  Result := DetermineEntityData(ClassNameColl, ClassBodyColl);
end;

function TPHPCustomClass.GetMethodCount: Integer;
begin
  Result := FMethods.Count;
end;

function TPHPCustomClass.GetMethods(Index: Integer): TPHPClassMethod;
begin
  Result := TPHPClassMethod(FMethods[Index]);
end;

function TPHPCustomClass.GetMethodsEx(Index: Integer): TPHPMethod;
begin
  Result := GetMethods(Index);
end;

function TPHPCustomClass.GetVariableCount: Integer;
begin
  Result := FVariables.Count;
end;

function TPHPCustomClass.GetVariables(Index: Integer): TPHPVariable;
begin
  Result := TPHPVariable(FVariables[Index]);
end;

{$IFNDEF PHP4}
function TPHPCustomClass.GetConstantCount: Integer;
begin
  Result := FConstants.Count;
end;

function TPHPCustomClass.GetConstants(Index: Integer): TPHPClassConstant;
begin
  Result := TPHPClassConstant(FConstants[Index]);
end;
{$ENDIF}

procedure TPHPCustomClass.ParseEntity;
var
  MatchColl: IMatchCollection;
  {$IFDEF PARSEMETHODBODY}
  ClassColl: IMatchCollection;
  MatchColl2: IMatch;
  {$ENDIF}
  i: integer;
begin
  FVariables.Clear;
  FMethods.Clear;
  {$IFNDEF PHP4}
  FConstants.Clear;
  {$ENDIF}
  if FOccurence <> '' then
  begin
    FComments.Source := FOccurence;
    with VarRegEx do
    begin
      MatchColl := Matches(FOccurence);
      for i := 0 to MatchColl.Count-1 do
        if not FComments.CheckIsCommented(MatchColl.Items[i].Index,
          MatchColl.Items[i].Length) then
          FVariables.Add(TPHPVariable.Create(Self, MatchColl.Items[i].Value,
            FTextPos + MatchColl.Items[i].Index, MatchColl.Items[i]));
    end;
    {$IFNDEF PHP4}
    with ConstantRegEx do
    begin
      MatchColl := Matches(FOccurence);
      for i := 0 to MatchColl.Count-1 do
        if not FComments.CheckIsCommented(MatchColl.Items[i].Index,
          MatchColl.Items[i].Length) then
          FConstants.Add(TPHPClassConstant.Create(Self,
            MatchColl.Items[i].Value, FTextPos + MatchColl.Items[i].Index,
            MatchColl.Items[i]));
    end;
    {$ENDIF}
    // search function in source files
    {$IFDEF PARSEMETHODBODY}
    with MethodRegEx do
    begin
      ClassColl := Matches(FOccurence);
      for i := 0 to ClassColl.Count-1 do
      begin
        if not FComments.CheckIsCommented(ClassColl.Items[i].Index,
          ClassColl.Items[i].Length) then 
        with ClassBodyRegEx do
        begin
          MatchColl2 := Match(FOccurence, ClassColl.Items[i].Index);
          if MatchColl2.Matched then
            FMethods.Add(TPHPClassMethod.Create(Self, ClassColl.Items[i].Value +
              MatchColl2.Value, FTextPos + ClassColl.Items[i].Index,
              ClassColl.Items[i], MatchColl2));
        end;
      end;
    end;
    {$ELSE}
    with MethodRegEx do
    begin
      MatchColl := Matches(FOccurence);
      for i := 0 to MatchColl.Count-1 do
      begin
        if (not FComments.CheckIsCommented(MatchColl.Items[i].Index,
          MatchColl.Items[i].Length)) then
          FMethods.Add(TPHPClassMethod.Create(Self, MatchColl.Items[i].Value,
            FTextPos + MatchColl.Items[i].Index, MatchColl.Items[i], nil));
      end;
    end;
    {$ENDIF}
  end;
end;

procedure TPHPCustomClass.ParseEntity(const Match: IMatch);
begin
  // Match parameter is ignored
  ParseEntity;
end;

procedure TPHPCustomClass.SetSourceText(const Value: string);
begin
  if FOccurence <> Value then
  begin
    FOccurence := Value;
    FName := DetermineEntityName(Occurence);
    ParseEntity;
  end;
end;

function TPHPCustomClass.FindEntityAt(APos: Integer): TPHPEntity;
var
  I: integer;
  Method: TPHPMethod;
  Variable: TPHPVariable;
  {$IFNDEF PHP4}
  Constant: TPHPClassConstant;
  {$ENDIF}
begin
  Result := nil;
  if (APos >= TextPos) and (APos <= Length(FOccurence) + TextPos) then
  begin
    i := 0;
    while (i < GetMethodCount) and (Result = nil) do
    begin
      // fix v2.4
      // just use one call (instead of 3 or more)
      Method := GetMethods(i);
      if (APos >= Method.TextPos)
      {$IFDEF PARSEMETHODBODY}
      and (APos <= Method.EndBracket) then
      {$ELSE}
      and (APos <= (Length(Method.Occurence) +
        GetMethods(i).TextPos)) then
      {$ENDIF}
        Result := Method;
      inc(i);
    end;
    if Result = nil then
    begin
      i := 0;
      while (i < GetVariableCount) and (Result = nil) do
      begin
        // fix v2.4
        // just use one call (instead of 3 or more)
        Variable := GetVariables(i);
        if (APos >= Variable.TextPos) and
          (APos <= (Length(Variable.Occurence) +
          GetVariables(i).TextPos)) then
          Result := Variable;
        inc(i);
      end;
    end;
    {$IFNDEF PHP4}
    if Result = nil then
    begin
      i := 0;
      while (i < GetConstantCount) and (Result = nil) do
      begin
        // fix v2.4
        // just use one call (instead of 3 or more)      
        Constant := GetConstants(i);
        if (APos >= Constant.TextPos) and
          (APos <= (Length(Constant.Occurence) +
          Constant.TextPos)) then
          Result := Constant;
        inc(i);
      end;
    end;
    {$ENDIF}
    if Result = nil then
      Result := Self;
  end;
end;

{$IFNDEF PHP4}
{ TPHPInterface }

constructor TPHPInterface.Create(Parent: TPHPCustomClass;
  const SourceText: string; TextPos: Integer);
begin
  inherited Create(Parent, SourceText, TextPos);
end;

constructor TPHPInterface.Create(Parent: TPHPCustomClass; const ClassNameMatch,
  ClassBodyMatch: IMatch; const SourceText: string; TextPos: Integer);
begin
  inherited Create(Parent, ClassNameMatch, ClassBodyMatch, SourceText, TextPos);
end;

function TPHPInterface.GetInterfaceMethod(
  Index: Integer): TPHPInterfaceMethod;
begin
  Result := TPHPInterfaceMethod(FMethods[Index]);
end;

function TPHPInterface.DetermineEntityData(const NameMatch,
  BodyMatch: IMatch): string;
begin
  if (NameMatch <> nil) and (BodyMatch <> nil) then
  begin
    if NameMatch.Matched then
    begin
      Result := NameMatch.Groups.Items[2].Value;
      if NameMatch.Groups.Items[6].Value = 'extends' then
        FExtends := NameMatch.Groups.Items[8].Value else
      FExtends := '';
    end else
    Result := '';
    if BodyMatch.Matched then
    begin
      FBeginBracket := FTextPos + BodyMatch.Index;
      FEndBracket := FTextPos + BodyMatch.Index + BodyMatch.Length;
    end;
  end;
end;

function TPHPInterface.DetermineEntityName(const Text: string): string;
var
  NameMatch, BodyMatch: IMatch;
begin
  with InterfaceNameRegEx do
    NameMatch := Match(Text);
  with ClassBodyRegEx do
    BodyMatch := Match(Text);
  Result := DetermineEntityData(NameMatch, BodyMatch);
end;

procedure TPHPInterface.ParseEntity;
var
  MatchColl: IMatchCollection;
  i: integer;
begin
  FMethods.Clear;
  FVariables.Clear;
  FConstants.Clear;
  if FOccurence <> '' then
  begin
    FComments.Source := FOccurence;
    // search constants
    with ConstantRegEx do
    begin
      MatchColl := Matches(FOccurence);
      for i := 0 to MatchColl.Count-1 do
        if not FComments.CheckIsCommented(MatchColl.Items[i].Index,
          MatchColl.Items[i].Length) then
          FConstants.Add(TPHPClassConstant.Create(Self,
            MatchColl.Items[i].Value, FTextPos + MatchColl.Items[i].Index,
            MatchColl.Items[i]));
    end;
    // search function in source files
    with ClassMethodRegEx do
    begin
      MatchColl := Matches(FOccurence);
      for i := 0 to MatchColl.Count-1 do
      begin
        // fix 1.7
        if (not FComments.CheckIsCommented(MatchColl.Items[i].Index,
          MatchColl.Items[i].Length)) then
          FMethods.Add(TPHPInterfaceMethod.Create(Self,
            MatchColl.Items[i].Value, FTextPos + MatchColl.Items[i].Index,
            MatchColl.Items[i], nil));
      end;
    end;
  end;
end;
{$ENDIF}

{ TPHPSource }

constructor TPHPSource.Create(const SourceText: string);
begin
  // create instances first, because ParseEntity is called in the inherited
  // constructor and this will cause a call to the overriden procedure that uses
  // these fields.
  FClasses := TObjectList.Create;
  {$IFNDEF PHP4}
  FInterfaces := TObjectList.Create;
  {$ELSE}
  FConstants := TObjectList.Create; 
  {$ENDIF}
  FIncludes := TObjectList.Create;
  FSyntaxErrors := TStringList.Create;
  FResultOutPut := TStringList.Create;
  FStrictPHPTagCheck := True;
  inherited Create(nil, SourceText, 0);
end;

constructor TPHPSource.Create(Parent: TPHPCustomClass; const Occurence: string;
  TextPos: Integer);
begin
  // TextPos and Parent parameters are ignored by this class
  Create(Occurence);
end;

destructor TPHPSource.Destroy;
begin
  FClasses.Free;
  {$IFNDEF PHP4}
  FInterfaces.Free;
  {$ELSE}
  FConstants.Free;
  {$ENDIF}
  FIncludes.Free;
  FSyntaxErrors.Free;
  FResultOutPut.Free;
  inherited;
end;

function TPHPSource.GetMethodsEx(Index: Integer): TPHPMethod;
begin
  Result := TPHPMethod(FMethods[index]);
end;

procedure TPHPSource.SetSourceText(const Value: string);
begin
  if Value <> FOccurence then
  begin
    FOccurence := Value;
    // fix v2.2
    // removed unnecessary call
    ParseEntity;
  end;
end;

{$IFDEF PHP4}
function TPHPSource.GetConstantCount: Integer;
begin
  Result := FConstants.Count;
end;
{$ENDIF}

function TPHPSource.GetConstantEx(Index: Integer): TPHPDefineConstant;
begin
  Result := TPHPDefineConstant(FConstants[Index]);
end;

function TPHPSource.GetClassCount: Integer;
begin
  Result := FClasses.Count;
end;

function TPHPSource.GetClasses(Index: Integer): TPHPClass;
begin
  Result := TPHPClass(FClasses[Index]);
end;

{$IFNDEF PHP4}
function TPHPSource.GetInterfaceCount: Integer;
begin
  Result := FInterfaces.Count;
end;

function TPHPSource.GetInterfaces(Index: Integer): TPHPInterface;
begin
  Result := TPHPInterface(FInterfaces[Index]);
end;
{$ENDIF}

function TPHPSource.GetIncludeCount: Integer;
begin
  Result := FIncludes.Count;
end;

function TPHPSource.GetIncludes(Index: Integer): TPHPIncludeFile;
begin
  Result := TPHPIncludeFile(FIncludes[Index]);
end;

procedure TPHPSource.ParseEntity;

  function IsClassFunction(AStartPos: Integer): Boolean; 
  var
    i: integer;
  begin
    Result := False;
    for i := 0 to ClassCount-1 do
    begin
      Result := (AStartPos > Classes[i].TextPos) and
        (AStartPos < Length(Classes[i].Occurence) + Classes[i].TextPos);
      if Result then Break;
    end;
    {$IFNDEF PHP4}
    if not Result then
      for i := 0 to InterfaceCount-1 do
      begin
        Result := (AStartPos > Interfaces[i].TextPos) and
          (AStartPos < Length(Interfaces[i].Occurence) + Interfaces[i].TextPos);
        if Result then Break;
      end;
    {$ENDIF}
  end;

var
  FileColl,
  TempColl   : IMatchCollection;
  ClassColl  : IMatchCollection;
  MatchColl  : IMatch;
  MatchColl2 : IMatchCollection;
  SourceRegEx: IRegEx;
  i, j, k,
  Index,
  LastIndex,
  Temp       : Integer;
  TagFound   : Boolean;
  Value,
  LastValue,
  Diff,
  TempText,
  Text       : string;
begin
  FClasses.Clear;
  FMethods.Clear;
  FIncludes.Clear;
  FConstants.Clear;
  FVariables.Clear;
  {$IFNDEF PHP4}FInterfaces.Clear;{$ENDIF}
  if (FOccurence <> '') then
  begin
    FComments.Source := FOccurence;
    // only match code in <? ?> or <?php ?> (in strict mode)
    if FStrictPHPTagCheck then
      SourceRegEx := StrictSourceRegEx else
    SourceRegEx := uPHPInspector.SourceRegEx;
    with SourceRegEx do
    begin
      FileColl := Matches(FOccurence);
      Index := -1;
      // fix v.2.4
      Text := '';
      for j := 0 to FileColl.Count-1 do
      begin
        // fix v.1.9
        // we need to test if we have not have taken the php tag of another
        // match
        if (Index > FileColl.Items[j].Index) or
          ((Index + Length(Value)) > (FileColl.Items[j].Index +
            Length(FileColl.Items[j].Value))) then Continue;
        LastIndex := Index;
        Index := FileColl.Items[j].Index;
        LastValue := Value;
        Value := FileColl.Items[j].Value;
        // fix v.1.9 check for "real" php tags
        // check if <? is correct, else search next real tag
        if FComments.CheckIsCommented(Index, 2) then
        begin
          TempColl := BeginSourceRegEx.Matches(FOccurence);
          TagFound := False;
          for k := 0 to TempColl.Count-1 do
            if (TempColl.Items[k].Index > Index) and
              (not FComments.CheckIsCommented(TempColl.Items[k].Index, 2)) then
            begin
              Delete(Value, 1, TempColl.Items[k].Index - Index);
              Index := TempColl.Items[k].Index;
              TagFound := True;
              Break;
            end;
          // fix v2.0
          // no other tag was found, so we continue with the next pair of
          // php tags
          if (not TagFound) then
            Continue;
        end;
        // check if there even is a ?> and if its correct (else search
        // next correct tag)
        if (FStrictPHPTagCheck) or
          (FileColl.Items[j].Groups.Items[5].Value = '?>') then
        begin
          Temp := Index + Length(Value);
          if FComments.CheckIsCommented(Temp, 2) then
          begin
            TagFound := False;
            TempColl := EndSourceRegEx.Matches(FOccurence);
            for k := 0 to TempColl.Count-1 do
              if (TempColl.Items[k].Index > Temp) and
                (not FComments.CheckIsCommented(TempColl.Items[k].Index,
                 2)) then
              begin
                Value := Value + Copy(FOccurence, Temp + 1,
                  TempColl.Items[k].Index - Temp);
                TagFound := True;
                Break;
              end;
            // fix v2.0
            // if no end-tag was found and we have the StrictPHPTagCheck
            // option turned on, the loop will be continued with the next pair
            // of php tags
            if (not TagFound) and (FStrictPHPTagCheck) then
              Continue;
          end;
        end;
        // fix v2.2
        // "replace" text between the php-tags with #32's while matching
        // so that the whole text is parsed
        if FileColl.Count >= 1 then
        begin
          // fix v2.4
          // check if there is something before the first tag!
          if j = 0 then
          begin
            // FileColl.Items[0].Index may be not correct (e.g. if the taq is
            // commented)
            if (Index > 0) then
            begin
              SetLength(TempText, FileColl.Items[0].Index);
              FillChar(TempText[1], Length(TempText) * SizeOf(TempText[1]),
                #32);
            end else
              TempText := '';
          end else
          begin
            SetLength(TempText, Index - (LastIndex + Length(LastValue)));
            FillChar(TempText[1], Length(TempText) * SizeOf(TempText[1]), #32);
          end;
          Text := Text + TempText + Value;          
        end else
          Text := Value;
      end;
      // -------------------------
      // search the "class name (extends classname)" first, then the content
      // fix v1.3
      with ClassNameRegEx do
      begin
        ClassColl := Matches(Text);
        for i := 0 to ClassColl.Count-1 do
        begin
          // fix v.1.4
          // check for comments !before! fetching the text of the class body!
          if FComments.CheckIsCommented(ClassColl.Items[i].Index,
            ClassColl.Items[i].Length) then Continue;
          {$IFDEF NODUPLICATES}
          MatchColl := SimpleClassNameRegEx.Match(ClassColl.Items[i].Value);
          if MatchColl.Matched then
            if FindClass(MatchColl.Groups.Items[2].Value) <> -1 then Continue;
          {$ENDIF}
          with ClassBodyRegEx do
          begin
            MatchColl := Match(Text, ClassColl.Items[i].Index);
            if (MatchColl.Matched) then
            begin
              // difference text between the 2 matches
              if (ClassColl.Items[i].Index + ClassColl.Items[i].Length) <
                (MatchColl.Index) then
                Diff := Copy(Text, ClassColl.Items[i].Index + 1 +
                  ClassColl.Items[i].Length, MatchColl.Index -
                  (ClassColl.Items[i].Index + ClassColl.Items[i].Length)) else
              Diff := '';
              FClasses.Add(TPHPClass.Create(Self, ClassColl.Items[i],
                MatchColl, ClassColl.Items[i].Value + Diff + MatchColl.Value,
                ClassColl.Items[i].Index));
            end;
          end;
        end;
      end;
      {$IFNDEF PHP4}
      with InterfaceNameRegEx do
      begin
        ClassColl := Matches(Text);
        for i := 0 to ClassColl.Count-1 do
        begin
          // fix v1.6
          // applying v.l.4 patch for interfaces
          if FComments.CheckIsCommented(ClassColl.Items[i].Index,
            ClassColl.Items[i].Length) then Continue;
          {$IFDEF NODUPLICATES}
          MatchColl := SimpleClassNameRegEx.Match(ClassColl.Items[i].Value);
          if MatchColl.Matched then
            if FindInterface(
              MatchColl.Groups.Items[2].Value) <> -1 then Continue;
          {$ENDIF}
          with ClassBodyRegEx do
          begin
            MatchColl := Match(Text, ClassColl.Items[i].Index);
            if (MatchColl.Matched) then
            begin
              // difference text between the 2 matches
              if (ClassColl.Items[i].Index + ClassColl.Items[i].Length) <
                (MatchColl.Index) then
                Diff := Copy(Text, ClassColl.Items[i].Index + 1 +
                  ClassColl.Items[i].Length, MatchColl.Index -
                  (ClassColl.Items[i].Index + ClassColl.Items[i].Length)) else
              Diff := '';
              FInterfaces.Add(TPHPInterface.Create(Self, ClassColl.Items[i],
                MatchColl, ClassColl.Items[i].Value + Diff + MatchColl.Value,
                ClassColl.Items[i].Index));
            end;
          end;
        end;
      end;
      {$ENDIF}
      // search methods in source file
      {$IFDEF PARSEMETHODBODY}
      with DefaultMethodRegEx do
      begin
        ClassColl := Matches(Text);
        for i := 0 to ClassColl.Count-1 do
        begin
          if IsClassFunction(ClassColl.Items[i].Index) or
            (FComments.CheckIsCommented(ClassColl.Items[i].Index,
            ClassColl.Items[i].Length)) then Continue;
          {$IFDEF NODUPLICATES}
          MatchColl := DefaultMethodRegEx.Match(ClassColl.Items[i].Value);
          if MatchColl.Matched then
            if FindMethod(nil,
              MatchColl.Groups.Items[3].Value) <> nil then Continue;
          {$ENDIF}
          with ClassBodyRegEx do
          begin
            MatchColl := Match(Text, ClassColl.Items[i].Index);
            if MatchColl.Matched then
              FMethods.Add(TPHPMethod.Create(Self,
                ClassColl.Items[i].Value + MatchColl.Value,
                ClassColl.Items[i].Index, ClassColl.Items[i],
                MatchColl))
          end;
        end;
      end;
      {$ELSE}
      with MethodRegEx do
      begin
        MatchColl2 := Matches(Text);
        for i := 0 to MatchColl2.Count-1 do
        begin
          if (not IsClassFunction(MatchColl2.Items[i].Index)) and
            (not FComments.CheckIsCommented(MatchColl2.Items[i].Index,
             MatchColl2.Items[i].Length)) then
          begin
            {$IFDEF NODUPLICATES}
            MatchColl := DefaultMethodRegEx.Match(Copy(Text,
              MatchColl2.Items[i].Index, MatchColl2.Items[i].Length);
            if MatchColl.Matched then
              if FindMethod(nil,
                MatchColl.Groups.Items[3].Value) <> nil then Continue;
            {$ENDIF}
            FMethods.Add(TPHPMethod.Create(Self, MatchColl2.Items[i].Value,
              MatchColl2.Items[i].Index, MatchColl2.Items[i], nil));
          end,
        end;
      end;
      {$ENDIF}
      with IncludeRegEx do
      begin
        MatchColl2 := Matches(Text);
        for i := 0 to MatchColl2.Count-1 do
          if not FComments.CheckIsCommented(MatchColl2.Items[i].Index,
            MatchColl2.Items[i].Length) then
            FIncludes.Add(TPHPIncludeFile.Create(Self,
              MatchColl2.Items[i].Value, MatchColl2.Items[i].Index,
              MatchColl2.Items[i]));
      end;
      with DefineConstantRegEx do
      begin
        MatchColl2 := Matches(Text);
        for i := 0 to MatchColl2.Count-1 do
          if not FComments.CheckIsCommented(MatchColl2.Items[i].Index,
            MatchColl2.Items[i].Length) then
            FConstants.Add(TPHPDefineConstant.Create(Self,
              MatchColl2.Items[i].Value, MatchColl2.Items[i].Index,
              MatchColl2.Items[i]));
      end;
    end;
  end;
end;

class function TPHPSource.Version: string;
begin
  Result := 'PHP Inspector ' + PHPINSPECTORVERSION;
end;

function TPHPSource.SyntaxCheck(CheckForRuntimeErrors: Boolean = False;
  const FileName: string = ''): Boolean;
const
  ArgStr: array[Boolean] of string = ('-l', '-f');
  // -l (syntax check ) -> Syntax check, won't find fatal errors etc.
  // -f (parse and run) -> Will find runtime errors, too.
var
  TempFile,
  PHPFile  : string;
  RetCode  : Cardinal;
begin
  if (FileExists(FPHPExe)) then
  begin
    // create temporary file (if necessary)
    if (FileName = '') or (not FileExists(FileName)) then
    begin
      TempFile := CreateTempFile;
      if not FileExists(TempFile) then
      begin
        Result := True;
        Exit;
      end;
      with TStringList.Create do
      try
        Text := FOccurence;
        SaveToFile(TempFile);
      finally
        Free;
      end;
    end else TempFile := '';
    if TempFile <> '' then
      PHPFile := TempFile else
    PHPFile := FileName;
    FSyntaxErrors.Clear;
    FResultOutPut.Clear;
    try
      if RunCaptured(ExtractFilePath(PHPFile), FPHPExe,
        ArgStr[CheckForRuntimeErrors] + ' "' + PHPFile + '"', FSyntaxErrors,
        RetCode) then
      begin
        // no error -> code was parsed successfully and output is returned.
        if RetCode = 0 then
        begin
          Result := True;
          FResultOutPut.Assign(FSyntaxErrors);
          FSyntaxErrors.Clear;
          Exit;
        end;
        case FSyntaxErrors.Count of
          0: Result := True;
          1: begin
               with RegExCreate('(?i)No syntax errors detected in (.*)',
                 [rcoSingleLine]) do
                 Result := IsMatch(FSyntaxErrors[0]);
             end;
          else Result := False;
        end;
      end else Result := True;
    finally
      if (TempFile <> '') and (FileExists(TempFile)) then
        Windows.DeleteFile(PChar(TempFile));
    end;
  end else Result := True;
end;

procedure TPHPSource.LoadFromFile(const FileName: string);
begin
  if FileExists(FileName) then
    with TStringList.Create do
    try
      LoadFromFile(FileName);
      Self.SourceText := Text;
    finally
      Free;
    end;
end;

function TPHPSource.FindClass(const ClassName: string): Integer;
var
  i: integer;
begin
  i := 0;
  Result := -1;
  while (i < ClassCount) and (Result = -1) do
  begin
    if Classes[i].Name = ClassName then
      Result := i;
    inc(i); // fix v2.0
  end;
end;

{$IFNDEF PHP4}
function TPHPSource.FindInterface(const InterfaceName: string): Integer;
var
  i: integer;
begin
  i := 0;
  Result := -1;
  while (i < InterfaceCount) and (Result = -1) do
  begin
    if Interfaces[i].Name = InterfaceName then
      Result := i;
    inc(i); // fix v2.0
  end;
end;
{$ENDIF}

function TPHPSource.FindMethod(AClass: TPHPCustomClass;
  const MethodName: string): TPHPMethod;
type
  TGetMethod = function(Index: Integer): TPHPMethod of object;
var
  i,
  Count,
  Index   : Integer;
  Method  : TGetMethod;
begin
  if (AClass <> nil) then
  begin
    Index := FindClass(AClass.Name);
    if Index = -1 then
    begin
      {$IFNDEF PHP4}
      Index := FindInterface(AClass.Name);
      if Index = -1 then
      begin
      {$ENDIF}
        Result := nil;
        Exit;
      {$IFNDEF PHP4}
      end else
      begin
        Count := InterfaceCount;
        Method := Interfaces[Index].GetMethodsEx;
      end;
      {$ENDIF}
    end else
    begin
      Count := ClassCount;
      Method := Classes[Index].GetMethodsEx;
    end;
  end else
  begin
    Method := GetMethodsEx;
    Count := MethodCount;
  end;
  i := 0;
  Result := nil;
  while (i < Count) and (Result = nil) do
  begin
    if Method(i).Name = MethodName then
      Result := Method(i);
    inc(i); // fix v2.0
  end;
end;

function TPHPSource.FindEntityAt(APos: Integer): TPHPEntity;
var
  i: integer;
  Include: TPHPIncludeFile;
begin
  Result := inherited FindEntityAt(APos);
  if Result = Self then
    Result := nil;
  if Result = nil then
  begin
    i := 0;
    while (i < IncludeCount) and (Result = nil) do
    begin
      Include := Includes[i];
      if (APos >= Include.TextPos) and
        (APos <= (Length(Include.Occurence) + Include.TextPos)) then
        Result := Include;
      inc(i);
    end;
    if Result = nil then
    begin
      i := 0;
      while (i < ClassCount) and (Result = nil) do
      begin
        Result := Classes[i].FindEntityAt(APos);
        inc(i);
      end;
      {$IFNDEF PHP4}
      if Result = nil then
      while (i < InterfaceCount) and (Result = nil) do
      begin
        Result := Interfaces[i].FindEntityAt(APos);
        inc(i);
      end;
      {$ENDIF}
    end;
  end;
end;

// ---------------------------------------------------------------------------
// Create RegExes to save time when parsing the source
var
  Comment, SingleComment, SingleQuote, DoubleQuote: string;

initialization
  ClassNameRegEx := RegexCreate(ABSTRACTCLASS + 'class(\s+?)(\w+?)' + CLASSSTR +
    '(\s*?)', [rcoUngreedy, rcoSingleLine]);
  ClassBodyRegEx := RegexCreate('\{(((?>[^{}]+)|(?R))*)\}', [rcoSingleLine]);
  DefaultMethodRegEx := RegExCreate('function(\s+?)(|\&)(\w*)(\s*?)\((.*)\)',
    [rcoSingleLine, rcoUngreedy]);
  MethodRegEx := RegexCreate(BuildMethodVisibility + BuildStatic +
    'function(\s+?)(\&|)(\w*)(\s*?)\((.*)\)', [rcoSingleLine, rcoUngreedy]);
  IncludeRegEx := RegExCreate('(?<![\$\w\x7f-\xff])(include_once|include|' +
    'require_once|require)(\s*)([\w\W]+)(?=;|$|\?)', [rcoMultiLine, rcoUngreedy]);
  StrictSourceRegEx := RegExCreate('\<\?(.*)\?\>', [rcoSingleLine,
    rcoUngreedy]);
  SourceRegEx := RegExCreate('(((\<\?)(.*)(\?\>))|(\<\?(.*?)))',
    [rcoSingleLine, rcoUngreedy]);
  BeginSourceRegEx := RegExCreate('\<\?', [rcoSingleLine, rcoUngreedy]);
  EndSourceRegEx   := RegExCreate('\?\>', [rcoSingleLine, rcoUngreedy]);
  DefineConstantRegEx := RegExCreate('define(\s*)\((?:\''|\")([\w\s]*)' +
    '(?:\''|\")(\s*),(\s*)(?:\''|\"|)(.[^\''\"]*)(?:\''|\"|)(\s*)(,(\s*)(.*)' +
    '(\s*)|)\)', [rcoUngreedy, rcoSingleLine]);
  MethodParamRegEx := RegExCreate('(\&|)\$(\w+?)(((\s*)\=(\s*)' +
    '((\")(.*)(\"))|(.*))|)(?=$|,)', [rcoUngreedy, rcoSingleLine]);
  {$IFDEF NODUPLICATES}
  SimpleClassNameRegEx := RegExCreate('class(\s+?)(\w+?)(\s*?)',
    [rcoUngreedy, rcoSingleLine]);
    {$IFNDEF PHP4}
    SimpleInterfaceNameRegEx := RegExCreate('interface(\s+?)(\w+?)(\s*?)',
      [rcoUngreedy, rcoSingleLine]);
    {$ENDIF}
  {$ENDIF}
  {$IFDEF FUNCTIONVARIABLES}
  VariableRegEx := RegExCreate('\$(\w+)');
  {$ENDIF}
  {$IFNDEF PHP4}
  InterfaceNameRegEx := RegExCreate('interface(\s+)(\w+?)(((\s+)(extends)' +
    '(\s+)(.*)(\s*)(?:\{))|)', [rcoSingleLine, rcoUngreedy]);
  ClassMethodRegEx := RegexCreate(BuildMethodVisibility(False) +
    'function(\s+?)(\&|)(\w*)(\s*?)\((.*)\)', [rcoSingleLine, rcoUngreedy]);
  VarRegEx := RegexCreate(VARSEARCHSTR + '(\s+?)' + BuildStatic +
    VARNAMESTR, [rcoSingleLine, rcoUngreedy]);
  ConstantRegEx := RegExCreate('const(\s+?)(\w*)(\s*)\=(\s*)(?:\''|\")(.*)' +
    '(?:\''|\")', [rcoSingleLine, rcoUngreedy]);
  {$ELSE}
  VarRegEx := RegexCreate(VARSEARCHSTR + '(\s+?)' + VARNAMESTR,
    [rcoSingleLine, rcoUngreedy]);
  {$ENDIF}
  // Comment RegEx 
  // fix v.1.4
  // old regexes:
  // until v.1.2 \/\*(((?>[^\*]+)|(?R))*)\*\/
  // until v.1.4 (/\*(?:[^*][^/])*\*/)
  // DoubleQuote := '(?:\"(?:\\.|[^\"\\])*\")';    <-- caused EStackOverflow !
  // SingleQuote := '(?:\''(?:\\.|[^\''\\])*\'')'; <-- caused EStackOverflow !
  // -----------------------------------------------------------------------
  // from http://aspn.activestate.com/ASPN/Cookbook/Rx/Recipe/59811
  Comment := '/\*[^*]*\*+([^/*][^*]*\*+)*/';
  // fix v.2.5
  // match unix-shell like comments, too
  SingleComment := '(\/\/|\#)(.*?)';
  // from http://www.regular-expressions.info/examplesprogrammer.html
  // fix v.2.5
  // do not match escaped characters
  DoubleQuote := '"(.*)(?<!\\)"';
  SingleQuote := '''(.*)(?<!\\)''';
  CommentRegEx := RegExCreate('(?:(?:' + DoubleQuote + '|' + SingleQuote +
    ')|((?:' + Comment + ')|(' + SingleComment + ')))', [rcoUngreedy]);
finalization
  ClassNameRegEx := nil;
  ClassBodyRegEx := nil;
  MethodRegEx := nil;
  IncludeRegEx := nil;
  VarRegEx := nil;
  CommentRegEx := nil;
  SourceRegEx := nil;
  StrictSourceRegEx := nil;
  BeginSourceRegEx := nil;
  EndSourceRegEx := nil;
  DefineConstantRegEx := nil;
  MethodParamRegEx := nil;
  {$IFDEF FUNCTIONVARIABLES}
  VariableRegEx := nil;
  {$ENDIF}
  {$IFDEF NODUPLICATES}
  SimpleClassNameRegEx := nil;
    {$IFNDEF PHP4}
    SimpleInterfaceNameRegEx := nil;
    {$ENDIF}
  {$ENDIF}
  {$IFNDEF PHP4}
  ConstantRegEx := nil;
  ClassMethodRegEx := nil;
  InterfaceNameRegEx := nil;
  {$ENDIF}
end.
